﻿unit find_email;

interface

uses
  Classes, Common, DataEntry, DataStorage, DateUtils, Regex, SysUtils, Variants;

const
  ATRY_EXCEPT_STR = 'TryExcept: '; //noslz
  RPAD_VALUE = 30;
  CHAR_LENGTH = 80;
  SPACE = ' ';
  TSWT = 'The script will terminate.';
  BLANK = '<BLANK>';

  SIZE_GREATER_THAN = 'Filter by Size:';
  TASKLIST_BETWEEN_DATES = 'Between Dates';
  TASKLIST_START_DATE = 'Start Date:';
  TASKLIST_END_DATE = 'End Date:';
  TASKLIST_HAS_ATTACHMENT = 'Has Attachment';  
  TASKLIST_MIN_SIXE = 'Min Size:';
  TASKLIST_MAX_SIZE = 'Max Size:';

var
  Report_StringList: TStringList;  

implementation

function ShellExecute(hWnd: cardinal; lpOperation: Pchar; lpFile: Pchar; lpParameter: Pchar; lpDirectory: Pchar; nShowCmd: integer): Thandle; stdcall; external 'Shell32.Dll' name 'ShellExecuteW';

//------------------------------------------------------------------------------
// Function: Right Pad v2
//------------------------------------------------------------------------------
function RPad(const AString: string; AChars: integer): string;
begin
  AChars := AChars - Length(AString);
  if AChars > 0 then
    Result := AString + StringOfChar(' ', AChars)
  else
    Result := AString;
end;

//------------------------------------------------------------------------------
// Function: Make sure the InList and OutList exist
//------------------------------------------------------------------------------
function ListExists(aList: TListOfEntries; ListName_str: string): boolean;
begin
  Result := False;
  if assigned(aList) then
  begin
    Progress.Log(RPad(ListName_str + SPACE + 'count:', rpad_value) + IntToStr(aList.Count));
    Result := True;
  end
  else
    Progress.Log(RPad('Not Assigned:', rpad_value) + ListName_str);
end;

//------------------------------------------------------------------------------
// Function: Parameters Received
//------------------------------------------------------------------------------
function ParametersReceived: integer;
var
  i: integer;
begin
  Result := 0;
  Progress.Log(Stringofchar('-', 80));
  if CmdLine.ParamCount > 0 then
  begin
    Result := CmdLine.ParamCount;
    Progress.Log(RPad('Params received:', RPAD_VALUE) + IntToStr(CmdLine.ParamCount));
    Progress.Log('Params:');
    for i := 0 to CmdLine.ParamCount - 1 do
    begin
      if not Progress.isRunning then break;
      Progress.Log(RPad(' Param ' + IntToStr(i) + ':', RPAD_VALUE) + cmdline.params[i]);
    end;

    // Add to Report_StringList
    if assigned(Report_StringList) then
    begin
      Report_StringList.Add(RPad('Params received:', RPAD_VALUE) + IntToStr(CmdLine.ParamCount));
      Report_StringList.Add('Params:');
      for i := 0 to CmdLine.ParamCount - 1 do
      begin
        Report_StringList.Add(RPad(' Param ' + IntToStr(i) + ':', RPAD_VALUE) + cmdline.params[i]);
        if not Progress.isRunning then break;
      end;
    end;

  end
  else
    Progress.Log('No parameters were received.');
  Progress.Log(Stringofchar('-', 80));
end;

//------------------------------------------------------------------------------
// Procedure: Create PDF
//------------------------------------------------------------------------------
procedure CreateRunTimePDF(aStringList: TStringList; SaveTitle: string; font_size_int: integer; dtInFileName: boolean);
const
  HYPHEN = ' - ';
var
  CurrentCaseDir: string;
  ExportDate: string;
  ExportDateTime: string;
  ExportedDir: string;
  ExportTime: string;
  SaveName: string;
begin
  sleep(1000);  // Required if you create reports in quick succession.
  CurrentCaseDir := GetCurrentCaseDir;
  ExportedDir := CurrentCaseDir + 'Reports\Run Time Reports\';
  ExportDate := formatdatetime('yyyy-mm-dd', now);
  ExportTime := formatdatetime('hh-nn-ss', now);
  ExportDateTime := ExportDate + ' - ' + ExportTime + HYPHEN;
  if dtInFileName = False then
    ExportDateTime := '';
  SaveName := ExportedDir + ExportDateTime + SaveTitle;
  if DirectoryExists(CurrentCaseDir) then
  begin
    try
      if ForceDirectories(IncludeTrailingPathDelimiter(ExportedDir)) and DirectoryExists(ExportedDir) then
      begin
        if GeneratePDF(SaveName, aStringList, font_size_int) then
          Progress.Log(RPad('Success Creating:', RPAD_VALUE) + ExtractFileName(SaveName));
      end
      else
        Progress.Log('Failed Creating: ' + SaveName);
    except
      Progress.Log(ATRY_EXCEPT_STR + 'Exception creating runtime folder');
    end;
  end
  else
    Progress.Log('There is no current case folder');
  Progress.Log(Stringofchar('-', 80));
end;

//------------------------------------------------------------------------------
// Procedure: Simple PDF
//------------------------------------------------------------------------------
procedure SimpleRunTimePDF(aString: string; save_name_str: string);
const
  HYPHEN = ' - ';
var
  CurrentCaseDir: string;
  ExportedDir: string;
  SaveName: string;
  aStringList: TStringList;
begin
  aStringList := TStringList.Create;
  try
    aStringList.Add(aString);
    CurrentCaseDir := GetCurrentCaseDir;
    ExportedDir := CurrentCaseDir + 'Reports\Run Time Reports\';
    SaveName := ExportedDir + save_name_str + '.pdf';
    if DirectoryExists(CurrentCaseDir) then
    begin
      try
        if ForceDirectories(IncludeTrailingPathDelimiter(ExportedDir)) and DirectoryExists(ExportedDir) then
        begin
          if GeneratePDF(SaveName, aStringList, 8) then
            Progress.Log('Success Creating: ' + SaveName)
        end
        else
          Progress.Log('Failed Creating: ' + SaveName);
      except
        Progress.Log(ATRY_EXCEPT_STR + 'Exception creating runtime folder');
      end;
    end
    else
      Progress.Log('There is no current case folder');
  finally
    aStringList.free;
  end;
end;

//------------------------------------------------------------------------------
// Procedure: Find
//------------------------------------------------------------------------------
function find(cmdparam_pos: integer; aString: string) : boolean;
var
  j: integer;
  LineList: TStringList;
begin
  Result := False;
  LineList := TStringList.Create;
  LineList.Delimiter := ',';
  LineList.StrictDelimiter := True; // Required: strings are separated *only* by Delimiter  
  try
  if assigned(linelist) and (CmdLine.ParamCount > cmdparam_pos) then
  begin
    lineList.Clear;
    if (trim(CmdLine.Params[cmdparam_pos]) = '') or (trim(CmdLine.Params[cmdparam_pos]) = 'NOPARAM') then
    begin
      Result := True;
    end
    else
    begin
      lineList.CommaText := CmdLine.Params[cmdparam_pos];
      for j := 0 to lineList.Count -1 do
      begin
        if not Progress.isRunning then 
          break;
        if trim(UpperCase(lineList[j])) <> 'NOPARAM' then
        begin
          if (POS(UpperCase(lineList[j]), UpperCase(aString)) > 0) then
          begin
            Result := True;
            //Report_StringList.Add('match: ' + lineList[j] + ' << with >> ' + aString);
          end;
        end;
      end;
    end;
  end
  finally
    LineList.free;
  end;
end;

//------------------------------------------------------------------------------
// Start of Script
//------------------------------------------------------------------------------
const
  POS0 = 0;
  POS1 = 1;
  POS2 = 2;
  
var
  anEntry: TEntry;
  bl_attachments_only: boolean;
  bl_between_dates: boolean;
  bl_Continue: boolean;
  bl_from_match: boolean;
  bl_subject_match: boolean;
  bl_to_match: boolean;
  col_Attachments: TDataStoreField;
  col_Received: TDataStoreField;
  col_SentFrom: TDataStoreField;
  col_SentTo: TDataStoreField;
  colResult: string;
  dsEmail: TDataStore;
  dt_received: TDateTime;
  end_date_dt: TDateTime;
  end_date_str: string;
  find_email_from_str: string;
  find_email_subject_str: string;
  find_email_to_str: string;
  i: integer;
  sent_from_str: string;
  sent_to_str: string;
  start_date_dt: TDateTime;
  start_date_str: string;
  subject_str: string;

begin
  // Read the parameters
  if ParametersReceived > 0 then
  begin
  end
  else
  begin
    SimpleRunTimePDF('Did not receive a search criteria.' + #13#10 +
    'In the "Select Profile" window, use the "Edit" option to enter the search terms.', 'Error - Did not receive a search criteria');  //aString: string; save_name_str: string
    Exit;
  end;

  // Check that InList and OutList exist, and InList count > 0
  if (not ListExists(InList, 'InList')) or (not ListExists(OutList, 'OutList')) then
  begin
    Progress.Log(Stringofchar('-', 80));
    Exit
  end
  else
  if InList.Count = 0 then
  begin
    Progress.Log('InList count is zero.' + SPACE + TSWT);
    Progress.Log(Stringofchar('-', 80));
    Exit;
  end
  else
    Progress.Log(Stringofchar('-', 80));

  dsEmail := GetDataStore(DATASTORE_EMAIL);
  if dsEmail = nil then
  begin
    Progress.Log('Email module returned nil' + SPACE + TSWT);
    Exit;
  end;
  
  Report_StringList := TStringList.Create;

  try
    // Set Columns
    col_SentTo := dsEmail.DataFields.GetFieldByName('TO'); //noslz
    col_SentFrom := dsEmail.DataFields.GetFieldByName('FROM'); //noslz
    col_Attachments := dsEmail.DataFields.FieldByName('HAS_ATTACHMENTS');

    //--------------------------------------------------------------------------
    // ScriptTask Variable: Attachments Only
    //--------------------------------------------------------------------------
    bl_attachments_only := False;    
    if RegexMatch(ScriptTask.GetTaskListOption(TASKLIST_HAS_ATTACHMENT), 'True', False) then
    begin
      bl_attachments_only := True;
      Progress.Log(RPad('Attachments Only:', rpad_value) + 'True');  
    end
    else
      Progress.Log(RPad('Attachments Only:', rpad_value) + 'False');
    
    //--------------------------------------------------------------------------
    // ScriptTask Variable: Between Dates
    //--------------------------------------------------------------------------
    bl_between_dates := False;
    if RegexMatch(ScriptTask.GetTaskListOption(TASKLIST_BETWEEN_DATES), 'True', False) then    
    begin
      // Start Date
      start_date_str := ScriptTask.GetTaskListOption(TASKLIST_START_DATE);
      if trim(start_date_str) = '' then
      begin      
        Progress.Log('Error reading input form - Date Start.');
        bl_between_dates := False;
      end
      else
      begin
        try
          start_date_dt := VarToDateTime(start_date_str);
          bl_between_dates := True;
        except
          Progress.Log(ATRY_EXCEPT_STR + 'Error converting Start Date string to DateTime: ' + start_date_str);
          bl_between_dates := False;
        end;
      end;

      // End Date
      end_date_str := ScriptTask.GetTaskListOption(TASKLIST_END_DATE);
      if trim(end_date_str) = '' then
      begin
        Progress.Log('Error reading input form - Date End.');
        bl_between_dates := False;
      end
      else if bl_between_dates = True then
      begin
        try
          end_date_dt := VarToDateTime(end_date_str);
        except
          Progress.Log(ATRY_EXCEPT_STR + 'Error converting End Date string to DateTime: ' + end_date_str);
          bl_between_dates := False;
        end;
      end;
    end;

    // Log the Between Dates Result
    if bl_between_dates then
    begin
      Progress.Log(RPad(TASKLIST_BETWEEN_DATES, rpad_value) + 'True');
      Progress.Log(RPad(SPACE + TASKLIST_START_DATE, rpad_value) + start_date_str);
      Progress.Log(RPad(SPACE + TASKLIST_END_DATE, rpad_value) + end_date_str);
    end
    else
      Progress.Log(RPad(TASKLIST_BETWEEN_DATES, rpad_value) + 'False');
    
    Progress.Log(Stringofchar('-', CHAR_LENGTH));

    //--------------------------------------------------------------------------
    // Loop the InList
    //--------------------------------------------------------------------------
    for i := 0 to InList.Count - 1 do
    begin
      if not Progress.isRunning then
        break;

      bl_Continue := True;

      // Initialize for each loop
      bl_subject_match := False;
      bl_from_match := False;
      bl_to_match := False;
      anEntry := TEntry(InList[i]);
      
      // Attachments -----------------------------------------------------------
      if bl_Continue and bl_attachments_only then
      begin
        if assigned(col_Attachments) then
        begin
          colResult := col_Attachments.AsString[anEntry];
          if (colResult <> 'Yes') then //noslz
            bl_Continue := False;
        end
        else
          Progress.Log('No attachments column');      
      end;
      
      // Between Dates ----------------------------------------------------------
      if bl_Continue and bl_between_dates then
      begin
        col_Received := dsEmail.DataFields.FieldByName('RECEIVED'); //noslz
        if assigned(col_Received) then
        begin
          dt_received := (col_Received.AsDateTime[anEntry]);
          if (CompareDate(dt_received, start_date_dt) = -1) or (CompareDate(dt_received, end_date_dt) = 1) then
            bl_Continue := False;        
        end;      
      end;
      
      // Keywords ---------------------------------------------------------------
      if bl_Continue and (CmdLine.ParamCount > 0) then
      begin
        subject_str := UpperCase(anEntry.EntryName);
        sent_from_str := UpperCase(col_SentFrom.AsString(anEntry));
        sent_to_str := UpperCase(col_SentTo.AsString(anEntry));

        // Search The Message
        bl_subject_match := find(POS0, subject_str);
        bl_from_match := find(POS1, sent_from_str);
        bl_to_match := find(POS2, sent_to_str);

        if bl_from_match and bl_to_match and bl_subject_match then
        begin
          bl_Continue := True;
        end
        else
          bl_Continue := False;
      end;

      if bl_Continue then
        OutList.Add(anEntry);

    end;
   
    //--------------------------------------------------------------------------
    // LOGGING: Get Subject, From and To strings from the form (as params)
    //--------------------------------------------------------------------------    
    find_email_subject_str := '';
    if CmdLine.ParamCount > 0 then
      find_email_subject_str := CmdLine.Params[0];
    if (trim(find_email_subject_str) = '') or (trim(find_email_subject_str) = 'NOPARAM') then
      find_email_subject_str := BLANK;
      
    find_email_from_str := '';
    if CmdLine.ParamCount > 1 then
      find_email_from_str := CmdLine.Params[1];
    if (trim(find_email_from_str) = '') or (trim(find_email_from_str) = 'NOPARAM') then
      find_email_from_str := BLANK;

    find_email_to_str := '';
    if CmdLine.ParamCount > 2 then
      find_email_to_str := CmdLine.Params[2];
    if (trim(find_email_to_str) = '') or (trim(find_email_to_str) = 'NOPARAM') then
      find_email_to_str := BLANK;     

    // Produce the Summary PDF
    Report_StringList.Add(Stringofchar('-', 80));
    Report_StringList.Add('Search Settings');
    Report_StringList.Add(Stringofchar('-', 80));
    Report_StringList.Add(RPad('Subject contains:', RPAD_VALUE) + find_email_subject_str);
    Report_StringList.Add(RPad('AND From contains:', RPAD_VALUE) + find_email_from_str);
    Report_StringList.Add(RPad('AND To contains:', RPAD_VALUE) + find_email_to_str);
    Report_StringList.Add(RPad('Must have Attachment:', RPAD_VALUE) + BoolToStr(bl_attachments_only, True));
    Report_StringList.Add(RPad(TASKLIST_BETWEEN_DATES, RPAD_VALUE) + BoolToStr(bl_between_dates, True));    
    if bl_between_dates then
    begin
      Report_StringList.Add(RPad(' Start Date:', rpad_value) + start_date_str);
      Report_StringList.Add(RPad(' End Date:', rpad_value) + end_date_str);    
    end;
    Report_StringList.Add(Stringofchar('-', 80));
    Report_StringList.Add(RPad('Messages Found:', RPAD_VALUE) + IntToStr(OutList.Count));

    CreateRunTimePDF(Report_StringList, 'Search Settings.pdf', 8, False); // True = UseDT

  finally
    dsEmail.free;
    Report_StringList.free;
  end;

  Progress.Log(Stringofchar('-', CHAR_LENGTH));
  Progress.Log(RPad('OutList on finish:', rpad_value) + IntToStr(OutList.Count));
  Progress.Log('Script finished.');
  
  if RegexMatch(ScriptTask.GetTaskListOption('OPENLOG'), 'True', False) then
    ShellExecute(0, nil, 'explorer.exe', pchar(Progress.LogFilename), nil, 1);

  if RegexMatch(ScriptTask.GetTaskListOption('OPENLOG'), 'True', False) then
    ShellExecute(0, nil, 'explorer.exe', pchar(Progress.LogFilename), nil, 1);
    
end.
